{{/* vim: set filetype=mustache: */}}
{{/*
Expand the name of the chart.
*/}}
{{- define "kublrCommon.name" -}}
    {{- default .Chart.Name .Values.nameOverride | trunc 32 | trimSuffix "-" -}}
{{- end -}}

{{/*
Create a default fully qualified app name.
We truncate at 32 chars because some Kubernetes name fields are limited to this (by the DNS naming spec).
If release name contains chart name it will be used as a full name.
*/}}
{{- define "kublrCommon.fullname" -}}
{{- if .Values.fullnameOverride -}}
    {{- .Values.fullnameOverride | trunc 32 | trimSuffix "-" -}}
{{- else -}}
    {{- $name := default .Chart.Name .Values.nameOverride -}}
    {{- if contains $name .Release.Name -}}
        {{- .Release.Name | trunc 32 | trimSuffix "-" -}}
    {{- else -}}
        {{- printf "%s-%s" .Release.Name $name | trunc 32 | trimSuffix "-" -}}
    {{- end -}}
{{- end -}}
{{- end -}}

{{- define "kublrCommon.servicename" -}}
{{- $defaultServiceName := printf "%s-%s" .Release.Name .Chart.Name -}}
{{- $servicename := default $defaultServiceName .Values.serviceNameOverride -}}
{{- printf "%s" $servicename | trunc 32 | trimSuffix "-" -}}
{{- end -}}

{{/*
Generate the full image name for a container.

Parameters:
- .image.name: The name of the image repository (e.g., "prom/alertmanager").
- .image.tag: The tag for the image (e.g., "v0.27.0").
- .image.registry: The registry for the image (e.g., "docker.io"). The default registry for the image. If not provided, an empty string is assumed, indicating that the "default" Docker Hub registry is used.
- .overrideRegistry: A key used to override the default registry based on mappings defined in `.Values.global.registry`.
- .overrideName: An optional parameter that, if present, specifies an alternative image name defined within `.image`. This allows dynamic selection of the image name based on certain conditions.

Example Usage:
    - name: alertmanager
      image: {{ include "kublrCommon.images" (merge ( dict "image" .Values.alertmanager.image "overrideRegistry" "docker_io" ) . ) }}
*/}}

{{- define "kublrCommon.images" -}}
{{- $repositoryName := .image.name -}}
{{- $tag := .image.tag -}}
{{- $overrideRegistry := .overrideRegistry -}}
{{- $registry := .image.registry | default "" -}}

{{- if hasKey . "overrideName" -}}
    {{- if hasKey .image (index . "overrideName") -}}
        {{- $repositoryName = index .image (index . "overrideName") -}}
    {{- end -}}
{{- end -}}

{{- $globalRegistry := $.Values.global.kublrConfig.dockerRegistry.override | default dict -}}

{{- if hasKey $globalRegistry $overrideRegistry -}}
    {{- $registry = index $globalRegistry $overrideRegistry -}}
{{- else if hasKey $globalRegistry "default" -}}
    {{- $registry = index $globalRegistry "default" -}}
{{- end -}}

{{- if $registry -}}
    {{ printf "%s/%s:%s" $registry $repositoryName $tag }}
{{- else -}}
    {{ printf "%s:%s" $repositoryName $tag }}
{{- end -}}
{{- end -}}

{{/*
This helper function merges two dictionaries into one. If there are conflicting keys, the value from the second dictionary takes precedence.

Usage:
{{ include "kublrCommon.mergeDicts" (list <dict1> <dict2>) }}

Example Usage:
dict1:
  key1: "value1"
  key2: "value2"

dict2:
  key2: "new-value2"
  key3: "value3"

{{ include "kublrCommon.mergeDicts" (list dict1 dict2) }}
Result:
key1: "value1"
key2: "new-value2"
key3: "value3"
*/}}
{{- define "kublrCommon.mergeDicts" -}}
{{- $dict1 := index . 0 -}}
{{- $dict2 := index . 1 -}}
{{- $result := dict -}}
{{- range $key, $value := $dict1 }}
    {{- if not (hasKey $dict2 $key) }}
        {{- $_ := set $result $key $value }}
    {{- end }}
{{- end }}
{{- range $key, $value := $dict2 }}
    {{- $_ := set $result $key $value }}
{{- end }}
{{- toYaml $result }}
{{- end }}

{{/*
Retrieve a value from a Kubernetes Secret in the current namespace.

Parameters:
- .Namespace: The namespace where the Secret is located.
- .Name: The name of the Secret resource.
- .Key: The key within the Secret's data from which to retrieve the value.

Usage:
This function can be used to fetch a specific secret value when configuring container environments, volumes, or any other Kubernetes resource that needs secret data.

Example Usage:
env:
  - name: DATABASE_PASSWORD
    valueFrom:
      secretKeyRef:
        name: {{ include "kublrCommon.getSecretVal" (dict "Namespace" .Release.Namespace "Name" "my-secret" "Key" "db-password") }}

Notes:
- This function leverages the Helm `lookup` function, which requires Tiller (Helm v2) or at least Helm v3.1.0.
- It returns an empty string if the secret or the specific key within the secret does not exist.
*/}}

{{- define "kublrCommon.getSecretVal" -}}
{{- index (((lookup "v1" "Secret" .Namespace .Name ) | default dict ).data | default dict ) .Key | default "" -}}
{{- end -}}

{{/*
Return an existing secret key or generate a random password value.

Parameters:
- .Namespace: The namespace where the Secret is located (inherited by "kublrCommon.getSecretVal").
- .Name: The name of the Secret resource (inherited by "kublrCommon.getSecretVal").
- .Key: The key within the Secret's data to retrieve (inherited by "kublrCommon.getSecretVal").
- .Length: The desired length of the generated password. Defaults to 16 if not specified.

Usage:
This function is particularly useful for generating default passwords for applications, while allowing for the possibility of overriding these values through predefined Secrets.

Example Usage:
rabbitmq-password: {{ ( default "" .Values.rabbitmqPassword | b64enc ) | default ( include "kublrCommon.getOrGeneratePass" ( dict "Namespace" .Release.Namespace "Name" $fullname "Key" "rabbitmq-password" "Length" 10 ) ) | quote }}
rabbitmq-erlang-cookie: {{ ( default "" .Values.rabbitmqErlangCookie | b64enc ) | default ( include "kublrCommon.getOrGeneratePass" ( dict "Namespace" .Release.Namespace "Name" $fullname "Key" "rabbitmq-erlang-cookie" "Length" 32 ) ) | quote }}
clients-password: {{ ( default "" .Values.rabbitmqClientsPassword  | b64enc ) | default ( include "kublrCommon.getOrGeneratePass" ( dict "Namespace" .Release.Namespace "Name" $fullname "Key" "clients-password" "Length" 32 ) ) | quote }}

Notes:
- This function uses the Helm built-in functions `randAlphaNum` for generating a random password and `b64enc` for base64-encoding it.
- The generated password is returned in base64-encoded form to facilitate its direct use as a Secret value.
- It is recommended to use this function with caution, especially in production environments, to ensure that secret management practices align with security policies.
*/}}

{{- define "kublrCommon.getOrGeneratePass" -}}
{{- $val := include "kublrCommon.getSecretVal" . -}}
{{- $length := .Length | default 16 -}}
{{- if ( eq $val "" ) -}}{{- $val = randAlphaNum $length | b64enc -}}{{- end -}}
{{- $val | toString -}}
{{- end -}}