{{/* 
Helper to determine the configuration context: platform, centralLogging, selfHosted, or platformClient. 

Example Usage:
context: {{ include "kublrCommon.configContext" . }}

This line would output platform, centralLogging, selfHosted, or platformClient based on which flag is enabled in the Helm chart's values file, 
helping to conditionally render other parts of the chart based on the deployment context.

{{- if eq (include "kublrCommon.configContext" .) "platform" -}}
  "4G"
{{- end -}}
*/}}
{{- define "kublrCommon.configContext" -}}
  {{- $contexts := list 
    (dict "key" "platform" "default" false) 
    (dict "key" "centralLogging" "default" false) 
    (dict "key" "selfHosted" "default" false) 
    (dict "key" "platformClient" "default" false) 
  -}}
  {{- $result := "" -}}
  {{- range $context := $contexts -}}
    {{- $config := index $.Values ($context.key) -}}  {{/* Retrieve the config map for the current context key */}}
    {{- if and $config (eq (default ($context.default) $config.enabled) true) -}}  {{/* Safely check if the config is enabled */}}
      {{- $result = $context.key -}}
      {{- break -}}
    {{- end -}}
  {{- end -}}
  {{- if ne $result "" -}}
    {{- printf $result -}}
  {{- end -}}
{{- end -}}

{{/*
Generalized helper for getting values based on the configuration context. Fetches a specified value from .Values,
considering the current configuration context (e.g., platform, centralLogging, selfHosted, platformClient, etc.).

Parameters:
- key: A string representing the hierarchical key within .Values to fetch, using dot notation (e.g., "alertmanager.enabled").
The key should not include the context prefix (e.g., "platform.", "selfHosted.") as the context is determined dynamically.

Example Usage:
alertmanager:
  enabled: {{- include "kublrCommon.getValue" (merge (dict "key" "alertmanager.enabled") . ) -}}

This line will fetch the alertmanager.enabled value from the .Values map, looking into the specific context determined 
(e.g., under .Values.platform, .Values.selfHosted, or .Values.platformClient).
*/}}
{{- define "kublrCommon.getValue" -}}
  {{- $context := include "kublrCommon.configContext" . -}}
  {{- $keyList := splitList "." .key -}}
  {{- $key := last $keyList -}}
  {{- $pathList := initial $keyList -}}
  {{- $value := index .Values $context -}}
  {{- $found := not (eq $value nil) -}}
  {{- range $pathElement := $pathList -}}
    {{- if and $found (hasKey $value $pathElement) -}}
      {{- $value = index $value $pathElement -}}
      {{- $found = not (eq $value nil) -}}
    {{- else -}}
      {{- $found = false -}}
    {{- end -}}
  {{- end -}}
  {{- if and $found (not (eq $value nil)) -}}
    {{- index $value $key | default "" -}}
  {{- end -}}
{{- end -}}

{{/*
Define a function to determine the Docker registry value.

Parameters:
- key: The registry key (e.g., "cr_kublr_com").
- overrides: A map of overrides.
- default: The default registry value if no override is found.

Example Usage:
global:
  {{- if and .Values.global.kublrConfig.dockerRegistry .Values.global.kublrConfig.dockerRegistry.override }}
  kublrConfig:
    dockerRegistry:
      override:
        cr_kublr_com: {{ include "kublrCommon.getRegistry" (list "cr_kublr_com" .Values.global.kublrConfig.dockerRegistry.override "cr.kublr.com") }}
        docker_io: {{ include "kublrCommon.getRegistry" (list "docker_io" .Values.global.kublrConfig.dockerRegistry.override "docker.io") }}
        quay_io: {{ include "kublrCommon.getRegistry" (list "quay_io" .Values.global.kublrConfig.dockerRegistry.override "quay.io") }}
        k8s_gcr_io: {{ include "kublrCommon.getRegistry" (list "k8s_gcr_io" .Values.global.kublrConfig.dockerRegistry.override "k8s.gcr.io") }}
        registry_k8s_io: {{ include "kublrCommon.getRegistry" (list "registry_k8s_io" .Values.global.kublrConfig.dockerRegistry.override "registry.k8s.io") }}
  {{- end }}

rabbitmq:
  image:
    registry: {{ include "kublrCommon.getRegistry" (list "docker_io" (index (default dict .Values.global.kublrConfig.dockerRegistry) "override") (default "docker.io" .Values.rabbitmq.image.registry)) }}
  exporter:
    image:
      registry: {{ include "kublrCommon.getRegistry" (list "docker_io" (index (default dict .Values.global.kublrConfig.dockerRegistry) "override") (default "docker.io" .Values.rabbitmq.exporter.image.registry)) }}

nodeGeneralValues: &nodeGeneralValues
  extraInitContainers: |
    - name: opensearch-init
      image: "{{ include "kublrCommon.getRegistry" (list "cr_kublr_com" .Values.global.kublrConfig.dockerRegistry.override "docker.kublr-dev.com") }}/kublr/opensearch-init:2.8.0"
*/}}

{{- define "kublrCommon.getRegistry" -}}
  {{- $key := index . 0 -}}
  {{- $overrides := index . 1 -}}
  {{- $default := index . 2 -}}
  {{- if and $overrides (index $overrides $key) -}}
    {{- index $overrides $key -}}
  {{- else if and $overrides (index $overrides "default") -}}
    {{- index $overrides "default" -}}
  {{- else -}}
    {{- $default -}}
  {{- end -}}
{{- end -}}

{{- define "kublrCommon.controlPlaneDomain" -}}
  {{- if and .Values.global.kublrConfig.controlPlaneEndpoints .Values.global.kublrConfig.controlPlaneEndpoints.domain -}}
    {{- .Values.global.kublrConfig.controlPlaneEndpoints.domain -}}
  {{- else -}}
    "*"
  {{- end -}}
{{- end -}}

{{- define "kublrCommon.controlPlaneNamespace" -}}
  {{- default "kublr" .Values.controlPlaneNamespace -}}
{{- end -}}

{{- define "kublrCommon.controlPlaneReleaseName" -}}
  {{- default "kcp" .Values.controlPlaneReleaseName -}}
{{- end -}}