# Kublr's Helm Framework
Helm is a powerful package manager for Kubernetes. It enables developers and operators to package, share, and deploy applications and services onto Kubernetes clusters. Helm uses a packaging format called charts. A chart is a collection of files that describe a related set of Kubernetes resources. The Kublr's Helm Framework is designed to standardize and simplify the creation of Helm charts, ensuring consistency, reusability, and maintainability. With the provided set of helper templates and functions, chart developers can easily define and deploy a variety of Kubernetes resources.

## Resource Templates and Components

- [Core Kubernetes Templates](#core-kubernetes-templates)
    - [ServiceAccount](#crsa)
    - [Role & ClusterRole](#crr)
    - [RoleBinding & ClusterRoleBinding](#crrb)
    - [Ingress](#cri)
    - [Labels](#cl)
- [Container Templates](#container-templates)
    - [keycloak-init](#ck-init)
    - [oauth2-proxy](#co2-proxy)
- [Hooks Templates](#hooks-templates)
    - [Update Resource](#cur)
- [Functions](#functions)
    - [kublrCommon.mergeDicts](#mergedicts)
    - [kublrCommon.images](#kublrcommonimages)
- [Values](#values)
    - [API Versions](#api-versions)
    - [Global Images](#global-images)
    - [Global Ingress Annotations](#global-ingress-annotations)
- [Values Template Generator](#values-template-generator)
- [Best Practices](#bp)
    - [Adding `kublr-common` as a Dependency](#bp-adding-framework-as-a-dependency)
    - [Creating New Helm Charts Using `app-template`](#bp-creating-new-helm-charts)
    - [Commenting Resources](#bp-commenting-resources)
    - [Merging with Root Context](#bp-merging-with-root-context)
    - [Naming Conventions](#naming-conventions)

## Core Kubernetes Templates <a id="core-kubernetes-templates"></a>

## `kublrCommon.render.serviceAccount` <a id="crsa"></a>

The `kublrCommon.render.serviceAccount` is a Helm helper template function that renders a Kubernetes ServiceAccount resource with customizable properties. You can leverage this function to maintain consistency in ServiceAccount generation across different charts or services.

## Table of Contents

- [Parameters](#crsa-parameters)
    - [appName](#crsa-appname)
    - [namespace](#crsa-namespace)
    - [imagePullSecretName](#crsa-imagepullsecretname)
    - [labels](#crsa-labels)
    - [annotations](#crsa-annotations)
- [Example Usage](#crsa-example-usage)

## Parameters <a id="crsa-parameters"></a>

### `appName` <a id="crsa-appname"></a>

- **Description**: The name of the application for which the ServiceAccount is being created.
- **Type**: String
- **Required**: Yes

### `namespace` <a id="crsa-namespace"></a>

- **Description**: The namespace in which the ServiceAccount will be created.
- **Type**: String
- **Default**: Value of `.Release.Namespace`
- **Required**: No

### `imagePullSecretName` <a id="crsa-imagepullsecretname"></a>

- **Description**: Specifies an optional image pull secret to be associated with the ServiceAccount. This is useful for pulling images from private registries.
- **Type**: String
- **Default**: Value of `.Values.imagePullSecretName`
- **Required**: No

### `labels` <a id="crsa-labels"></a>

- **Description**: Custom labels to be added to the ServiceAccount. These will be merged with default/common labels.
- **Type**: Dictionary
- **Default**: None
- **Required**: No

### `annotations` <a id="crsa-annotations"></a>

- **Description**: Annotations to be added to the ServiceAccount.
- **Type**: Dictionary
- **Default**: None
- **Required**: No

## Example Usage <a id="crsa-example-usage"></a>

To use the `kublrCommon.render.serviceAccount` function, you can integrate the following code:

```yaml
{{- include "kublrCommon.render.serviceAccount" (merge (dict 
  "appName" "prometheus" 
) . ) }}
```
## `kublrCommon.render.role` <a id="crr"></a>

The `kublrCommon.render.role` function is a Helm helper template that renders either a Kubernetes `Role` or `ClusterRole` resource, based on the specified parameters. This allows users to define permissions, using the Role-Based Access Control (RBAC) system in Kubernetes, for applications and services.

### Table of Contents

- [Parameters](#crr-parameters)
    - [appName](#crr-appname)
    - [isClusterRole](#crr-isclusterrole)
    - [rules](#crr-rules)
    - [namespace](#crr-namespace)
    - [roleNamePrefix](#crr-rolenameprefix)
    - [roleNameSuffix](#crr-rolenamesuffix)
    - [annotations](#crr-annotations)
- [Example Usage](#crr-example-usage)

### Parameters <a id="crr-parameters"></a>

#### `appName` <a id="crr-appname"></a>

- **Description**: The name of the application for which the Role or ClusterRole is being created.
- **Type**: String
- **Required**: Yes

#### `isClusterRole` <a id="crr-isclusterrole"></a>

- **Description**: Determines if the rendered resource should be a `ClusterRole` (if set to `true`) or a `Role` (if set to `false` or not provided).
- **Type**: Boolean
- **Default**: `false`
- **Required**: No

#### `rules` <a id="crr-rules"></a>

- **Description**: A list of rules defining the permissions for the Role or ClusterRole.
- **Type**: List of Dictionaries
- **Required**: Yes

#### `namespace` <a id="crr-namespace"></a>

- **Description**: The namespace in which the Role will be created. Not applicable if `isClusterRole` is `true`.
- **Type**: String
- **Default**: Value of `.Release.Namespace`
- **Required**: No

#### `roleNamePrefix` <a id="crr-rolenameprefix"></a>

- **Description**: An optional prefix to the generated name for the Role or ClusterRole.
- **Type**: String
- **Default**: None
- **Required**: No

#### `roleNameSuffix` <a id="crr-rolenamesuffix"></a>

- **Description**: An optional suffix to the generated name for the Role or ClusterRole.
- **Type**: String
- **Default**: None
- **Required**: No

#### `annotations` <a id="crr-annotations"></a>

- **Description**: Annotations to be added to the Role or ClusterRole.
- **Type**: Dictionary
- **Default**: None
- **Required**: No

### Example Usage <a id="crr-example-usage"></a>

To use the `kublrCommon.render.role` function, you can integrate the following code:

```yaml
{{- $prometheusRules := list 
  (dict 
    "apiGroups" (list "")
    "resources" (list "nodes" "nodes/proxy" "services" "endpoints" "pods")
    "verbs"     (list "get" "list" "watch")
  )
  (dict 
    "apiGroups" (list "extensions" "networking")
    "resources" (list "ingresses")
    "verbs"     (list "get" "list" "watch")
  )
  (dict 
    "nonResourceURLs" (list "/metrics")
    "verbs"           (list "get")
  )
  (dict 
    "apiGroups" (list "")
    "resources" (list "secrets")
    "verbs"     (list "*")
  )
}}
{{- include "kublrCommon.render.role" (merge (dict 
  "appName" "prometheus" 
  "isClusterRole" true 
  "rules" $prometheusRules
) . ) }}
```

## `kublrCommon.render.roleBinding` <a id="crrb"></a>

The `kublrCommon.render.roleBinding` function renders a Helm helper template that produces either a `RoleBinding` or a `ClusterRoleBinding` resource in Kubernetes. This function is highly customizable, ensuring consistent generation of these binding resources across different Helm charts or services.

### Table of Contents

- [Parameters](#crrb-parameters)
    - [appName](#crrb-appname)
    - [isClusterRole](#crrb-isclusterrole)
    - [namespace](#crrb-namespace)
    - [bindingNamePrefix](#crrb-bindingnameprefix)
    - [bindingNameSuffix](#crrb-bindingnamesuffix)
    - [annotations](#crrb-annotations)
    - [subjectName](#crrb-subjectname)
    - [roleRefName](#crrb-rolerefname)
    - [roleRefApiGroup](#crrb-rolerefapigroup)
    - [roleRefKind](#crrb-rolerefkind)
    - [extraSubjects](#crrb-extrasubjects)
- [Example Usage](#crrb-example-usage)

### Parameters <a id="crrb-parameters"></a>

#### `appName` <a id="crrb-appname"></a>
- **Description**: Name of the application.
- **Type**: String
- **Required**: Yes

#### `isClusterRole` <a id="crrb-isclusterrole"></a>
- **Description**: Determines if the rendered resource should be a `ClusterRole` (if set to `true`) or a `Role` (if set to `false` or not provided).
- **Type**: Boolean
- **Default**: `false`
- **Required**: No

#### `namespace` <a id="crrb-namespace"></a>
- **Description**: Namespace where the `RoleBinding` will be created.
- **Type**: String
- **Default**: `.Release.Namespace`
- **Required**: No

#### `bindingNamePrefix` <a id="crrb-bindingnameprefix"></a>
- **Description**: Prefix for the binding's name.
- **Type**: String
- **Default**: None
- **Required**: No

#### `bindingNameSuffix` <a id="crrb-bindingnamesuffix"></a>
- **Description**: Suffix for the binding's name.
- **Type**: String
- **Default**: None
- **Required**: No

#### `annotations` <a id="crrb-annotations"></a>
- **Description**: Annotations to add to the `RoleBinding` or `ClusterRoleBinding`.
- **Type**: Dictionary
- **Default**: Empty dictionary
- **Required**: No

#### `subjectName` <a id="crrb-subjectname"></a>
- **Description**: Name for the `ServiceAccount` subject.
- **Type**: String
- **Default**: `".Release.Name"-"appName"`
- **Required**: No

#### `roleRefName` <a id="crrb-rolerefname"></a>
- **Description**: Reference name for the role being bound.
- **Type**: String
- **Default**: `".Release.Name"-"appName"`
- **Required**: No

#### `roleRefApiGroup` <a id="crrb-rolerefapigroup"></a>
- **Description**: API group for the role reference.
- **Type**: String
- **Default**: `"rbac.authorization.k8s.io"`
- **Required**: No

#### `roleRefKind` <a id="crrb-rolerefkind"></a>
- **Description**: Kind of role reference (either "Role" or "ClusterRole").
- **Type**: String
- **Default**: Determined by `isClusterRole`
- **Required**: No

#### `extraSubjects` <a id="crrb-extrasubjects"></a>
- **Description**: Additional subjects to include in the binding.
- **Type**: List of Dictionaries
- **Default**: Empty list
- **Required**: No

### Example Usage <a id="crrb-example-usage"></a>

```yaml
{{- include "kublrCommon.render.roleBinding" (merge (dict 
  "appName" "prometheus" 
  "isClusterRole" true 
) . ) }}
```

## `kublrCommon.render.ingress` <a id="cri"></a>

The kublrCommon.render.ingress function is a Helm helper template that renders a Kubernetes Ingress resource. This function offers extensive customization options, enabling users to adapt and override nearly all properties of the rendered Ingress.

### Table of Contents

- [Parameters](#cri-parameters)
    - [appName](#cri-appName)
    - [oidcPort](#cri-oidcPort)
    - [defaultPort](#cri-defaultPort)
    - [path](#cri-path)
    - [annotations](#cri-annotations)
    - [labels](#cri-labels)
    - [ingressClassName](#cri-ingressClassName)
    - [globalIngressHosts](#cri-globalIngressHosts)
    - [appIngressHosts](#cri-appIngressHosts)
    - [authentication](#cri-authentication)
    - [serviceName](#cri-serviceName)
- [Example Usage](#cri-example-usage)

### Parameters <a id="cri-parameters"></a>

#### `appName` <a id="cri-appName"></a>
- **Description**: Specifies the name of the application for which the ingress is created.
- **Type**: String
- **Default**: None
- **Required**: Yes

#### `oidcPort` <a id="cri-oidcPort"></a>
- **Description**: Port to be used when OIDC authentication is enabled.
- **Type**: Integer
- **Default**: None
- **Required**: Yes

#### `defaultPort` <a id="cri-defaultPort"></a>
- **Description**: Default port for the ingress.
- **Type**: Integer
- **Default**: None
- **Required**: Yes

#### `path` <a id="cri-path"></a>
- **Description**: Path for the ingress rule.
- **Type**: String
- **Default**: Constructed using / and appName (e.g., /appName)
- **Required**: No

#### `annotations` <a id="cri-annotations"></a>
- **Description**: Annotations to be added to the ingress resource.
- **Type**: Dictionary
- **Default**: Default values from global.ingress.defaultAnnotations, with exclusions from global.ingress.defaultAnnotationsExclude
- **Required**: No

#### `labels` <a id="cri-labels"></a>
- **Description**: Labels to be added to the ingress resource.
- **Type**: String
- **Default**: From common.labels function
- **Required**: No

#### `ingressClassName` <a id="cri-ingressClassName"></a>
- **Description**: Specifies the ingress class name.
- **Type**: Dictionary
- **Default**: None
- **Required**: No

#### `globalIngressHosts` <a id="cri-globalIngressHosts"></a>
- **Description**: Global hosts for the ingress.
- **Type**: String or Map
- **Default**: None
- **Required**: Yes

#### `appIngressHosts` <a id="cri-appIngressHosts"></a>
- **Description**: Application-specific hosts for the ingress.
- **Type**: String or Map
- **Default**: None
- **Required**: Yes

#### `authentication` <a id="cri-authentication"></a>
- **Description**: Configuration related to authentication for the ingress.
- **Type**: Dictionary
- **Default**: None
- **Required**: Yes

#### `serviceName` <a id="cri-serviceName"></a>
- **Description**: Name of the service to be used in the ingress backend. If not provided, it defaults to a combination of the release name and appName.
- **Type**: String
- **Default**: None
- **Required**: No

## Example Usage <a id="cri-example-usage"></a>

Below is an example of how you can use the `kublrCommon.render.ingress` function:

```yaml
{{- $grafana := dict 
  "appName" "grafana" 
  "ingressClassName" .Values.grafana.ingress.ingressClassName 
  "globalIngressHosts" .Values.global.ingress.hosts 
  "appIngressHosts" .Values.grafana.ingress.hosts 
  "authentication" .Values.grafana.authentication 
  "oidcPort" 9180 
  "defaultPort" 3000 
  "annotations" (dict "helm.sh/hook" "post-install,post-upgrade" "helm.sh/hook-weight" "0") 
-}}
{{- include "kublrCommon.render.ingress" (merge $grafana .) }}
```

## `kublrCommon.labels` <a id="cl"></a>

The kublrCommon.labels function is a Helm helper template that renders a standard set of Kubernetes labels. These labels are commonly used to identify and filter resources associated with a specific Helm release, chart version, and more.
### Table of Contents

- [Parameters](#cl-parameters)
    - [appName](#cl-appName)
- [Example Usage](#cl-example-usage)

### Parameters <a id="cl-parameters"></a>

#### `appName` <a id="cl-appName"></a>
- **Description**: Specifies the name of the application for which the labels are generated.
- **Type**: String
- **Default**: None
- **Required**: Yes

## Example Usage <a id="cl-example-usage"></a>

Below is an example of how you can use the `kublrCommon.labels` function:

```helm
{{- include "kublrCommon.labels" (merge (dict "appName" "prometheus") . ) | nindent 4 }}
```

## Hooks Templates <a id="hooks-templates"></a>

## `common.update.resource` <a id="cur"></a>

The `common.update.resource` is a Helm helper template function designed to assist in updating Kubernetes resources when there's a change in their selectors. This function facilitates a seamless update process for Kubernetes resources like Deployments, DaemonSets, and StatefulSets.

### Table of Contents

- [Parameters](#cur-parameters)
    - [hookResourceName](#cur-hookResourcename)
    - [hookResourceType](#cur-hookResourcetype)
    - [hookExpectedLabels](#cur-hookexpectedlabels)
- [Example Usage](#cur-example-usage)

### Parameters <a id="cur-parameters"></a>

#### `hookResourceName` <a id="cur-hookResourcename"></a>

- **Description**: The name of the resource, which will be suffixed to the Helm release name to create the full resource name in the cluster.
- **Type**: String
- **Required**: Yes

#### `hookResourceType` <a id="cur-hookResourcetype"></a>

- **Description**: The type of the Kubernetes resource that needs to be updated. Supported resource types are `Deployment`, `DaemonSet`, and `StatefulSet`.
- **Type**: String
- **Required**: Yes
- **Possible Values**: `"Deployment"`, `"DaemonSet"`, `"StatefulSet"`

#### `hookExpectedLabels` <a id="cur-hookexpectedlabels"></a>

- **Description**: A dictionary of key-value pairs representing the labels that are expected to be present on the resource. If the actual labels on the resource differ, the resource will be deleted, and the Helm release process will create a new one.
- **Type**: Dictionary
- **Required**: Yes

### Example Usage: <a id="cur-example-usage"></a>

The following example demonstrates how to use the `common.update.resource` function:

```helm
{{- $scriptParams := merge (dict 
  "hookResourceName" "kibana" 
  "hookResourceType" "Deployment" 
  "hookExpectedLabels" (dict 
    "app" (printf "%s-%s" .Release.Name "kibana")
  )
) . -}}
{{- $script := include "common.update.resource" $scriptParams -}}
```

## Container Templates <a id="container-templates"></a>

## `common.keycloak-init` <a id="ck-init"></a>

A Helm helper template function designed for Keycloak initialization, facilitating the integration of the Keycloak authentication server with various applications.

### Table of Contents

- [Parameters](#ck-init-parameters)
    - [appName](#ck-init-appName)
- [Overriding Arguments](#ck-init-overriding-args)
- [Example Usage](#ck-init-example-usage)

### Parameters <a id="ck-init-parameters"></a>

#### `appName` <a id="ck-init-appName"></a>

- **Description**: Specifies the name of the application for which Keycloak authentication is being set up.
- **Type**: String
- **Required**: Yes


### Overriding Keycloak Parameters <a id="ck-init-overriding-args"></a>

- **Description**: Customize the Keycloak initialization parameters for each application using the `extraArgs`, `image`, and additional configurations like `extraEnv`, `extraEnvFrom`, and `extraVolumeMounts` within the application-specific section in the `values.yaml` file.

#### Arguments Customization
- **Type**: Dictionary
- **Usage**: Add argument keys (without the leading --) and values to the extraArgs dictionary.

#### Container Image Customization
- **Type**: Dictionary
- **Usage**: Specify the Docker image details to override the global default image.

#### Additional Configurations Customization
- **Type**: List of dictionaries
- **Usage**: Define additional environment variables, their sources, and volume mounts.

```yaml
kibana:
  authentication:
    keycloak:
      init:
        # Arguments customization
        extraArgs:
          keycloak-server: 'http://your-keycloak-server'
          keycloak-server-endpoint: '/auth'
          keycloak-admin-username: 'keycloak-admin'
          keycloak-admin-password: 'your-secret-password'
        # Container image customization
        image:
          registry: custom.registry.com
          name: custom/keycloak-init
          tag: 'latest'
        # Additional configurations customization
        extraEnv:
          - name: LOG_LEVEL
            value: "info"
        extraEnvFrom:
          - secretRef:
              name: my-keycloak-secret
        extraVolumeMounts:
          - name: keycloak-init-config
            mountPath: /etc/keycloak-init
            readOnly: true
```

#### Global Configuration Overrides for Keycloak Init
- **Description**: For consistent Keycloak init settings across all applications in the Chart, define global overrides within the `global.keycloak.init` section of the `values.yaml` file.

```yaml
global:
  keycloak:
    init:
      extraArgs:
        keycloak-server: 'http://global-keycloak-server'
        keycloak-server-endpoint: '/auth'
        keycloak-admin-username: 'admin'
        keycloak-admin-password: 'global-secret'
      image:
        registry: global.registry.com
        name: global/keycloak-init
        tag: 'latest'
      extraEnv:
        - name: GLOBAL_KEYCLOAK_LOG_LEVEL
          value: "info"
```

### Example Usage: <a id="ck-init-example-usage"></a>

The following example demonstrates how to use the `common.keycloak-init` function:

```helm
{{- include "common.keycloak-init" (merge (dict "appName" "prometheus") . ) | nindent 6 }}
```

## `common.oauth2-proxy` <a id="co2-proxy"></a>

The `common.oauth2-proxy` Helm helper template function is designed to facilitate the integration of the OAuth2 proxy with various applications, using OpenID Connect (OIDC) as an authentication provider. This function helps in generating the appropriate deployment configuration for the OAuth2 proxy sidecar container.

### Table of Contents

- [Parameters](#co2-proxy-parameters)
    - [appName](#co2-proxy-appname)
    - [path](#co2-proxy-path)
    - [prefix](#co2-proxy-prefix)
    - [upstreamPort](#co2-proxy-upstreamport)
    - [passAuthorizationHeader](#co2-proxy-passauthorizationheader)
    - [skipAuthRoute](#co2-proxy-skipauthroute)
- [Overriding Arguments](#co2-proxy-overriding-args)
- [Example Usage](#co2-proxy-example-usage)

### Parameters <a id="co2-proxy-parameters"></a>

#### `appName` <a id="co2-proxy-appname"></a>

- **Description**: The name of the application, used to determine specific configuration values for OIDC, such as `clientId` and `realm`.
- **Type**: String
- **Required**: Yes

#### `path` <a id="co2-proxy-path"></a>

- **Description**: The path at which the application is hosted and requires authentication via the OAuth2 proxy.
- **Type**: String
- **Required**: Yes

#### `prefix` <a id="co2-proxy-prefix"></a>

- **Description**: The path prefix for the OAuth2 proxy endpoints.
- **Type**: String
- **Default**: Uses the path and "/oauth2" in the format: '$path/oauth2'
- **Required**: No

#### `upstreamPort` <a id="co2-proxy-upstreamport"></a>

- **Description**: The port on which the main application (that is being protected by the OAuth2 proxy) is running.
- **Type**: String
- **Required**: Yes

#### `passAuthorizationHeader` <a id="co2-proxy-passauthorizationheader"></a>

- **Description**: Specifies whether to pass the authorization header to the upstream server. Defaults to `false` if not provided.
- **Type**: Boolean
- **Default**: false
- **Required**: No

#### `skipAuthRoute` <a id="co2-proxy-skipauthroute"></a>

- **Description**: A list of routes for which authentication via OAuth2 proxy should be skipped. These are appended to the specified path.
- **Type**: List of Strings
- **Required**: No

### Overriding Arguments <a id="co2-proxy-overriding-args"></a>

- **Description**: Customize the OAuth2 Proxy settings for each application by specifying `extraArgs`, `image`, and additional configurations like `extraEnv`, `extraEnvFrom`, and `extraVolumeMounts` within the application-specific configuration in the `values.yaml` file.

#### Arguments Customization
- **Type**: Dictionary
- **Usage**: Add argument keys (without the leading --) and values to the `extraArgs` dictionary. These arguments will override the default settings during the Helm chart rendering process.

#### Container Image Customization
- **Type**: Dictionary
- **Usage**: Define the Docker image details to use a custom image instead of the global default.

#### Additional Configurations Customization
- **Type**: List of dictionaries
- **Usage**: Specify additional environment variables, their sources, and volume mounts for the OAuth2 Proxy container.

```yaml
kibana:
  authentication:
    oauth2:
      # Arguments customization
      extraArgs:
        set-xauthrequest: 'false'
        show-debug-on-error: 'false'
        args-from-values: '{{ .Values.somevalue }}'
      # Container image customization
      image:
        registry: custom.registry.com
        name: custom/oauth2-proxy
        tag: v7.2.0
      # Additional configurations customization
      extraEnv:
        - name: LOG_LEVEL
          value: "debug"
      extraEnvFrom:
        - secretRef:
            name: my-secret
      extraVolumeMounts:
        - name: proxy-config
          mountPath: /etc/oauth2-proxy
          readOnly: true

```

#### Global Configuration Overrides for OAuth2 Proxy
- **Description**: Apply common OAuth2 Proxy configurations across all applications in the Chart using the `global.oauth2` section in the `values.yaml` file.

```yaml
global:
  oauth2:
    extraArgs:
      set-xauthrequest: 'false'
      show-debug-on-error: 'false'
    image:
      registry: global.registry.com
      name: global/oauth2-proxy
      tag: v7.2.1
    extraEnv:
      - name: GLOBAL_OAUTH2_LOG_LEVEL
        value: "info"
```

### Example Usage <a id="co2-proxy-example-usage"></a>

The following example demonstrates how to use the `common.oauth2-proxy` function:

```helm
{{- include "common.oauth2-proxy" (merge (dict 
      "appName" .Values.prometheus.authentication.oauth2.proxy.appName
      "path" .Values.prometheus.authentication.oauth2.proxy.path
      "skipAuthRoute" .Values.prometheus.authentication.oauth2.proxy.skipAuthRoute
      "upstreamPort" .Values.prometheus.authentication.oauth2.proxy.upstreamPort
  ) . ) | nindent 6 }}
```

## Functions <a id="functions"></a>

## `mergeDicts` <a id="mergedicts"></a>

A Helm helper template function designed to merge two dictionaries into one unified dictionary. In the event of conflicting keys between the two dictionaries, the function will prioritize the value from the second dictionary.

### Table of Contents

- [Parameters](#mergedicts-parameters)
    - [dict1](#mergedicts-dict1)
    - [dict2](#mergedicts-dict2)
- [Example Usage](#mergedicts-example-usage)

### Parameters <a id="mergedicts-parameters"></a>

#### `dict1` <a id="mergedicts-dict1"></a>

- **Description**: This represents the first dictionary to be merged.
- **Type**: Dictionary
- **Required**: Yes

#### `dict2` <a id="mergedicts-dict2"></a>

- **Description**: This is the second dictionary to be merged. If there are keys in this dictionary that conflict with keys in `dict1`, the values from `dict2` will be used in the merged result.
- **Type**: Dictionary
- **Required**: Yes

### Example Usage <a id="mergedicts-example-usage"></a>
```helm
dict1:
  key1: "value1"
  key2: "value2"

dict2:
  key2: "new-value2"
  key3: "value3"

{{ include "mergeDicts" (list dict1 dict2) }}
```

### Result
```helm
key1: "value1"
key2: "new-value2"
key3: "value3"
```

## `kublrCommon.images` <a id="kublrcommonimages"></a>

A Helm helper template function designed to generate the full image name for a container. It considers the repository name, tag, and registry, allowing for flexibility and customization.

### Table of Contents

- [Parameters](#kublrcommonimages-parameters)
    - [image](#kublrcommonimages-image)
    - [overrideRegistry](#kublrcommonimages-overrideRegistry)
    - [overrideName](#kublrcommonimages-overrideName)
- [Example Usage](#kublrcommonimages-example-usage)

### Parameters <a id="kublrcommonimages-parameters"></a>

#### `image` <a id="kublrcommonimages-image"></a>

- **Description**: Defines the container image details.
- **Type**: Dictionary
- **Required**: Yes

#### `overrideRegistry` <a id="kublrcommonimages-overrideRegistry"></a>

- **Description**: Overrides the image registry if specified.
- **Type**: string
- **Required**: Yes

#### `overrideName` <a id="kublrcommonimages-overrideName"></a>

- **Description**: Overrides the repository name if specified.
- **Type**: string
- **Required**: No

### Example Usage <a id="kublrcommonimages-example-usage"></a>
```helm
image:
  registry: "cr.kublr.com"
  name: "/kublr/base/kubectl"
  tag: "23.28.3"

{{ include "kublrCommon.images" (dict "image" .Values.image "overrideRegistry" "cr_kublr_com") }}
```

### Result
```helm
cr.kublr.com/kublr/base/kubectl:23.28.3
```
## Values <a id="values"></a>

### API Versions <a id="api-versions"></a>

It's beneficial to manage API versions for Kubernetes resources centrally in the `values.yaml` file. This practice ensures that if there's a need to shift to a new API version, it can be managed from a single location instead of updating individual templates.

```helm
apiVersions:
  deployment: "apps/v1"
  daemonSet: "apps/v1"
  statefulSet: "apps/v1"
  service: "v1"
  # ... [other apiVersions]
```

### Global Images <a id="global-images"></a>

Using a global configuration for images and resources is a useful way to maintain a centralized definition for frequently used images, which promotes consistency and reusability across the charts.

```helm
global:
  kubectlImage:
    registry: cr.kublr.com
    name: kublr/base/kubectl
    tag: "{{VERSION_KUBECTL}}"
    pullPolicy: IfNotPresent
    resources:
      requests:
        cpu: 30m
        memory: 100Mi
      limits:
        cpu: 200m
        memory: 200Mi
```

### Global Ingress Annotations <a id="global-ingress-annotations"></a>

Within the `values.yaml` file, you can define a set of default ingress annotations under the `global.ingress.defaultAnnotations` key. These annotations will apply to all `Ingresses`, unless overridden in a specific resource.

```helm
global:
  ingress:
    defaultAnnotations:
      ingress.kubernetes.io/custom-http-errors: "404,403,503"
      ingress.kubernetes.io/force-ssl-redirect: "true"
      ingress.kubernetes.io/proxy-buffer-size: "24k"
      # ... [other annotations]

```

#### Overriding Global Annotations

To override any of the global ingress annotations for a specific ingress resource, you can provide the desired annotations when calling the kublrCommon.render.ingress function.

```helm
ingress.kubernetes.io/proxy-buffer-size: "24k"
```

#### Removing Specific Annotations

To exclude specific default annotations from being applied to a particular Ingress resource, list them under global.ingress.defaultAnnotationsExclude in your values.yaml file. This will ensure that these annotations are not included for that Ingress resource.

```helm
global:
  ingress:
    defaultAnnotationsExclude:
      ingress.kubernetes.io/custom-http-errors: ""
      another.annotation.to.exclude: ""
```
## Values Template Generator <a id="values-template-generator"></a>

Logging: Values from operator for centralLogging 
```helm
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: kublr-system
      dockerRegistry: null
      NetworkDNSDomain: cluster.local
    controlPlaneEndpoints:
      baseURL: https://af35274f0739e4779b7d1311ae67601b-1089054637.us-east-1.elb.amazonaws.com
      apiEndpoint: https://af35274f0739e4779b7d1311ae67601b-1089054637.us-east-1.elb.amazonaws.com
      authEndpoint: https://af35274f0739e4779b7d1311ae67601b-1089054637.us-east-1.elb.amazonaws.com/auth/admin/kublr-ui/console
      authTokenEndpoint: https://af35274f0739e4779b7d1311ae67601b-1089054637.us-east-1.elb.amazonaws.com
      uiEndpoint: https://af35274f0739e4779b7d1311ae67601b-1089054637.us-east-1.elb.amazonaws.com
    kubeAPIEndpoint: https://100.64.0.1:443

centralLogging:
  clientNodes: 1
  dataNodes: 1
  enabled: true
  isPlatform: true
  logstashReplicas: 1
  masterNodes: 1
  persistent: true
  size: 8
```

Logging: Values from operator for selfHosted
```
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: cluster-space
      dockerRegistry: null
      networkDNSDomain: cluster.local
    controlPlaneEndpoints: {}
    kubeAPIEndpoint: https://100.64.0.1:443

selfHosted:
  clientNodes: 1
  dataNodes: 1
  enabled: true
  masterNodes: 1
  persistent: true
  size: 8G
```

Logging: Values from operator for Centralized Logging Agent
```
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: default
      dockerRegistry: null
      networkDNSDomain: cluster.local
    controlPlaneEndpoints: {}
    kubeAPIEndpoint: https://100.64.0.1:443

centralLogging:
  clientNodes: 1
  dataNodes: 1
  enabled: true
  isPlatform: false
  logstashReplicas: 1
  masterNodes: 1
  persistent: true
  size: 8G
```

Monitoring: Values from operator for platform
```
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: kublr-system
      dockerRegistry: null
      networkDNSDomain: cluster.local
    controlPlaneEndpoints:
      baseURL: https://ab5223b05caf4428c847977638eee3b8-824066613.us-east-1.elb.amazonaws.com
      apiEndpoint: https://ab5223b05caf4428c847977638eee3b8-824066613.us-east-1.elb.amazonaws.com
      authEndpoint: https://ab5223b05caf4428c847977638eee3b8-824066613.us-east-1.elb.amazonaws.com/auth/admin/kublr-ui/console
      authTokenEndpoint: https://ab5223b05caf4428c847977638eee3b8-824066613.us-east-1.elb.amazonaws.com
      uiEndpoint: https://ab5223b05caf4428c847977638eee3b8-824066613.us-east-1.elb.amazonaws.com
    kubeAPIEndpoint: https://100.64.0.1:443

platform:
  alertmanager:
    enabled: true
    persistent: true
    size: 2G
  enabled: true
  grafana:
    enabled: true
    persistent: true
    size: 2G
  prometheus:
    persistent: true
    size: 8G
```

Monitoring: Values from operator for selfHosted
```
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: cluster-space
      dockerRegistry: null
      networkDNSDomain: cluster.local
    controlPlaneEndpoints: {}
    kubeAPIEndpoint: https://100.64.0.1:443

selfHosted:
  alertmanager:
    enabled: true
    persistent: false
    size: ""
  enabled: true
  grafana:
    enabled: true
    persistent: false
    size: ""

```

Monitoring: Values from operator for Centralized Monitoring Agent
```
global:
  kublrConfig:
    imagePullSecretName: nexus.build
    dockerConfig:
      registry: docker.kublr-dev.com
      user: ecp-user
      password: pass
    helmConfig:
      repository: https://repo.kublr-dev.com/repository/helm
    clusterConfig:
      name: cluster-name
      space: default
      dockerRegistry: null
      networkDNSDomain: cluster.local
    controlPlaneEndpoints: {}
    kubeAPIEndpoint: https://100.64.0.1:443

platformClient:
  enabled: true
```

## Best Practices <a id="bp"></a>

### Adding kublr-common as a Dependency <a id="bp-adding-framework-as-a-dependency"></a>

For new Helm charts, it's essential to include the `kublr-common` chart as a dependency. This common chart provides a set of standard configurations and templates that ensure consistency and standardization across all charts.

#### Update Chart.yaml

Include the `kublr-common` chart as a dependency in the `Chart.yaml` file of your new chart with alias `common`.

```helm
dependencies:
  - name: kublr-common
    alias: common
    version: '{VERSION_KCP}'
    repository: file://../subcharts/kublr-common
```

#### Update BUILD.bazel

Ensure that `kublr-common` is added as a dependency in the `BUILD.bazel` file within the `helm_chart` function:

```helm
helm_chart(
    deps = [
        "//common:helm-chart",
    ],
)
```

### Creating New Helm Charts Using 'app-template' <a id="bp-creating-new-helm-charts"></a>
To create a new Helm Chart, copy the `app-template` directory to the location where the new application will store. The `app-template` directory serves as a base Helm chart containing the necessary resources that may be required for a new chart. `app-template` already has `Kublr's Helm Framework` included as a dependency. It is structured as follows:

```
app-template
├── app-helm-init.sh
├── BUILD.bazel
└── helm
    ├── Chart.yaml
    ├── templates
    │   ├── app_name-deployment.yaml
    │   ├── app_name-secret.yaml
    │   ├── chart-ingresses.yaml
    │   ├── chart-jobs.yaml
    │   ├── chart-rbac.yaml
    │   ├── chart-services.yaml
    │   ├── _helpers.tpl
    │   └── kublr
    │       └── migration-preupgrade.yaml
    └── values.yaml
```

#### Initialization Script: app-helm-init.sh
The `app-helm-init.sh` script is used to initialize a new Helm Chart by renaming all instances of `app_name` in the template files to the name of the new application. It also configures `BUILD.bazel` to build the Chart.

#### Usage
The script automatically derives the name from the directory and formats it appropriately.
```
./app-helm-init.sh
```

### Commenting Resources <a id="bp-commenting-resources"></a>

For every resource defined within a Helm template, it's a good practice to add a comment that specifies the name and kind of the resource. This practice makes templates easier to read and understand, especially for developers who might be unfamiliar with the structure of the chart.

#### Example 

```helm
{{/*  Create {{ .Release.Name }}-kibana RoleBinding */}}
{{- include "kublrCommon.render.roleBinding" (merge (dict 
  "appName" "kibana" 
  "isClusterRole" false 
) . ) }}
```
This comment gives a quick insight into the resource that is about to be created, making the template more self-explanatory.

### Merging with Root Context <a id="bp-merging-with-root-context"></a>

When using the include function in Helm to render another template and pass values to it, it's essential to merge any local or new dictionary with the root context `(.)`. Merging with the root context ensures that the included template has access to all the necessary values and maintains consistency throughout the Helm chart.

#### Example 

```helm
{{/* Create {{ .Release.Name }}-kibana Ingress */}}
{{- $kibana := dict 
  "appName" "kibana" 
  "ingressClassName" .Values.kibana.ingress.ingressClassName 
  "globalIngressHosts" .Values.global.ingress.hosts 
  "appIngressHosts" .Values.kibana.ingress.hosts 
  "authentication" .Values.kibana.authentication 
  "oidcPort" 9180 
  "defaultPort" 5601 
-}}
{{- include "kublrCommon.render.ingress" (merge $kibana .) }}
```

By using the `merge` function, the specific dictionary with parameters `$kibana` is combined with the root context. This approach ensures that `kublrCommon.render.ingress` has access to all the values from the parent template plus the specific `$kibana` values.

### Naming Conventions <a id="naming-conventions"></a>

Proper naming conventions play a crucial role in maintaining clarity and consistency across Helm charts in Kublr's Helm Framework. Adhering to these conventions ensures that developers can easily navigate and understand the structure and contents of the charts.

#### Core Kubernetes Templates Naming

For Core Kubernetes Templates, the naming convention for template files is to prepend an underscore to the resource name, followed by .tpl. This approach distinguishes these files as templates and quickly identifies their associated resource.

#### Format: 
```helm
_<resource_name>.tpl
```

#### Example: 
```helm
_deployment.tpl
```

#### Hooks Directory

All hook-related templates should be placed within a directory named hooks. This separation ensures that hooks are easily identifiable and distinguished from other templates.

#### Format: 
```helm
hooks/_<hook_name>.tpl
```

#### Example: 
```helm
hooks/_update-resource.tpl
```

#### Container Templates Naming

Container-specific templates must follow a consistent naming scheme to ensure clarity and ease of maintenance.

#### Format: 
```helm
_container-<container_name>.tpl
```

#### Example: 
```helm
_container-auth2-proxy.tpl
```

#### Small Chart File Naming

For small charts where resources are minimal, it's advisable to group related resources together in single files. Here are some naming conventions to follow:

    RBAC related resources (ServiceAccount, Roles, RoleBindings, etc.): chart-rbac.yaml
    Ingress resources: chart-ingresses.yaml
    Service resources: chart-services.yaml
