{{/* OAUTH2-PROXY

Example Usage:
{{- include "kublrCommon.oauth2-proxy" (merge (dict 
      "appName" "prometheus" 
      "path" "/prometheus" 
      "upstreamPort" "9090" 
  ) . ) | nindent 6 }}
*/}}
{{- define "kublrCommon.oauth2-proxy" }}
{{- $appName := .appName }}
{{- $path := .path }}
{{- $prefix := default (print $path "/oauth2") .prefix }}
{{- $upstreamPort := .upstreamPort }}
{{- $passAuthorizationHeader := default false .passAuthorizationHeader }}
{{- $globalOauth2 := default .Values.kublrCommon.global.oauth2 .Values.global.oauth2 -}}
{{- $oauthProxy := merge (index .Values $appName "authentication" "oauth2" | default dict) $globalOauth2 | deepCopy -}}
{{- $defaultArgs := dict 
    "http-address" "0.0.0.0:9180"
    "upstream" (print "http://127.0.0.1:" $upstreamPort)
    "provider-display-name" "Kublr"
    "client-id" (index .Values $appName "authentication" "oidc" "clientId")
    "client-secret" "fake-secret"
    "provider" "oidc"
    "oidc-issuer-url" (print "https://kcp-keycloak." .Release.Namespace ".svc:8443/auth/realms/" (index .Values $appName "authentication" "oidc" "realm"))
    "login-url" (print "/auth/realms/" (index .Values $appName "authentication" "oidc" "realm") "/protocol/openid-connect/auth")
    "redeem-url" (print "https://kcp-keycloak." .Release.Namespace ".svc:8443/auth/realms/" (index .Values $appName "authentication" "oidc" "realm") "/protocol/openid-connect/token")
    "profile-url" (print "https://kcp-keycloak." .Release.Namespace ".svc:8443/auth/realms/" (index .Values $appName "authentication" "oidc" "realm") "/protocol/openid-connect/userinfo")
    "validate-url" (print "https://kcp-keycloak." .Release.Namespace ".svc:8443/auth/realms/" (index .Values $appName "authentication" "oidc" "realm") "/protocol/openid-connect/userinfo")
    "ssl-insecure-skip-verify" "true"
    "insecure-oidc-skip-issuer-verification" "true"
    "cookie-secure" "true"
    "cookie-secret" "$(ENCRYPTION_KEY)"
    "cookie-path" $path
    "proxy-prefix" $prefix
    "set-xauthrequest" "true"
    "pass-access-token" "true"
    "pass-authorization-header" (toString $passAuthorizationHeader)
    "reverse-proxy" "true"
    "skip-provider-button" "true"
    "show-debug-on-error" "true"
    "oidc-email-claim" "sub"
    "scope" "openid email profile"
    "email-domain" "*"
    "allowed-group" "user"
    "oidc-groups-claim" "roles"
    "cookie-refresh" "15m"
}}
{{- /* Create a temporary dictionary to store processed extraArgs */ -}}
{{- $tempArgs := dict -}}
{{- range $key, $value := $oauthProxy.extraArgs -}}
  {{- /* Use tpl function to process the value as a template string */ -}}
  {{- $processedValue := tpl (toString $value) $ -}}
  {{- /* Update the temporary dictionary with the processed value */ -}}
  {{- $_ := set $tempArgs $key $processedValue -}}
{{- end -}}
{{- /* Merge the processed extraArgs with the defaultArgs */ -}}
{{- $finalArgs := merge $tempArgs $defaultArgs -}}

{{- /* Handling skip-auth-route argument */ -}}
{{- if .skipAuthRoute -}}
  {{- $formattedSkipAuthRoutes := list -}}
  {{- range $route := .skipAuthRoute -}}
    {{- $formattedRoute := print $path $route -}}
    {{- $formattedSkipAuthRoutes = append $formattedSkipAuthRoutes $formattedRoute -}}
  {{- end -}}
  {{- /* Join the formatted routes into a comma-separated string */ -}}
  {{- $skipAuthRouteStr := join "," $formattedSkipAuthRoutes -}}
  {{- /* Set the skip-auth-route in finalArgs */ -}}
  {{- $_ := set $finalArgs "skip-auth-route" $skipAuthRouteStr -}}
{{- end -}}

- name: oauth2-proxy
  image: {{ include "kublrCommon.images" (merge ( dict "image" $oauthProxy.image "overrideRegistry" "cr_kublr_com" ) . ) }}
  imagePullPolicy: {{ $oauthProxy.image.pullPolicy | default "IfNotPresent" }}
  args:
  {{- range $key, $value := $finalArgs }}
  - --{{ $key }}={{ $value }}
  {{- end }}
  ports:
    - containerPort: 9180
  env:
    - name: ENCRYPTION_KEY
      valueFrom:
        secretKeyRef:
          name: {{ .Release.Name }}-{{ $appName }}-jwks
          key: encryption-key
  {{- if $oauthProxy.extraEnv }}
    {{-  tpl (toYaml $oauthProxy.extraEnv) . | nindent 4 }}
  {{- end }}
  {{- if $oauthProxy.extraEnvFrom }}
  envFrom:
    {{- tpl (toYaml $oauthProxy.extraEnvFrom) . | nindent 4 }}
  {{- end }}
  {{- if $oauthProxy.extraVolumeMounts }}
  volumeMounts:
    {{- tpl (toYaml $oauthProxy.extraVolumeMounts) . | nindent 4 }}
  {{- end }}
{{- end }}